// pages/api/tickets/create.ts
import type { NextApiRequest, NextApiResponse } from "next";
import { z } from "zod";
import { prisma } from "@/lib/prisma";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/authOptions";
import {
  canonicalJsonHash,
  encodeCreateTicketArgs,
  walletClient,
  CONTRACT_ADDRESS,
  blockchainEnabled,
} from "@/lib/chain";

const CreateTicketSchema = z.object({
  title: z.string().min(3),
  description: z.string().min(3),
  location: z.string().optional().default(""),
  severity: z.number().int().min(0).max(5),

  departmentId: z.string().cuid().optional(),
  locationId: z.string().cuid().optional(),
  elementId: z.string().cuid().nullable().optional(),

  department: z.string().min(2).optional(),

  attachments: z.array(z.string()).optional().default([]),

  anonymousEmail: z.string().email().optional(),
  anonymousName: z.string().min(2).optional(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  if (req.method !== "POST")
    return res.status(405).json({ ok: false, error: "Method not allowed" });

  const session = await getServerSession(req, res, authOptions);

  try {
    const parsed = CreateTicketSchema.safeParse(req.body);
    if (!parsed.success) {
      return res.status(400).json({
        ok: false,
        error: "Invalid payload",
        details: parsed.error.flatten(),
      });
    }
    const hasSession = !!session?.user?.id;
    if (!hasSession && !parsed.data.anonymousEmail) {
      return res.status(400).json({
        ok: false,
        error: "Anonymous users must provide email address",
      });
    }

    const {
      title,
      description,
      location,
      severity,
      departmentId,
      locationId,
      elementId,
      department: legacyDepartmentInBody,
      attachments,
      anonymousEmail,
      anonymousName,
    } = parsed.data;

    let resolvedDepartmentName: string | null = null;
    let resolvedLocationId: string | null = null;

    if (departmentId) {
      const dep = await prisma.department.findUnique({
        where: { id: departmentId },
      });
      if (!dep) {
        return res
          .status(400)
          .json({ ok: false, error: "Invalid departmentId" });
      }
      resolvedDepartmentName = dep.name;

      if (locationId) {
        const loc = await prisma.location.findUnique({
          where: { id: locationId },
        });
        if (!loc) {
          return res
            .status(400)
            .json({ ok: false, error: "Invalid locationId" });
        }
        if (loc.departmentId !== dep.id) {
          return res.status(400).json({
            ok: false,
            error: "locationId does not belong to the selected departmentId",
          });
        }
        resolvedLocationId = locationId;

        if (elementId) {
          const el = await prisma.element.findUnique({
            where: { id: elementId },
          });
          if (!el) {
            return res
              .status(400)
              .json({ ok: false, error: "Invalid elementId" });
          }
          if (el.locationId !== loc.id) {
            return res.status(400).json({
              ok: false,
              error: "elementId does not belong to the selected locationId",
            });
          }
        }
      }
    } else {
      if (!legacyDepartmentInBody) {
        return res.status(400).json({
          ok: false,
          error: "Either departmentId or department (string) is required",
        });
      }
      resolvedDepartmentName = legacyDepartmentInBody;
    }

    const isLoggedIn = !!session?.user?.id;
    const creatorEmail = isLoggedIn
      ? session.user?.email ?? "unknown"
      : anonymousEmail ?? "anonymous";
    const creatorName = isLoggedIn
      ? session.user?.name ?? "Unknown User"
      : anonymousName ?? "Anonymous";

    const canonical = {
      title,
      description,
      location: location || "",
      severity,
      department: resolvedDepartmentName,
      departmentId: departmentId ?? null,
      locationId: resolvedLocationId ?? null,
      elementId: elementId ?? null,
      attachments,
      createdBy: creatorEmail,
      createdByName: creatorName,
      isAnonymous: !isLoggedIn,
      createdAt: new Date().toISOString(),
    };

    const jsonHash = canonicalJsonHash(canonical);

    const ticketData: any = {
      title,
      description,
      location: location || "",
      severity,
      department: resolvedDepartmentName ?? "",
      departmentId: departmentId ?? null,
      locationId: resolvedLocationId ?? null,
      elementId: elementId ?? null,

      attachmentsJson: JSON.stringify(attachments),
      canonicalJson: JSON.stringify(canonical),
    };

    if (isLoggedIn) {
      ticketData.createdById = (session.user as any).id;
    } else {
      ticketData.anonymousEmail = anonymousEmail;
      ticketData.anonymousName = anonymousName;
    }

    const saved = await prisma.ticket.create({
      data: ticketData,
      select: {
        id: true,
        title: true,
        description: true,
        location: true,
        severity: true,
        department: true,
        departmentId: true,
        locationId: true,
        elementId: true,
        createdAt: true,
        anonymousEmail: true,
        anonymousName: true,
      },
    });

    let txHash: string | null = null;
    let chainTicketId: number | null = null;

    if (blockchainEnabled && walletClient) {
      try {
        const data = encodeCreateTicketArgs(
          jsonHash,
          severity,
          resolvedDepartmentName || ""
        );

        txHash = await walletClient.sendTransaction({
          to: CONTRACT_ADDRESS,
          data,
        });

        await prisma.ticket.update({
          where: { id: saved.id },
          data: { txHash },
        });
      } catch (blockchainError: any) {
        console.error(
          "⚠️ Blockchain transaction failed:",
          blockchainError?.message || blockchainError
        );
        console.warn(
          "⚠️ Ticket created successfully but blockchain transaction failed - continuing without blockchain"
        );
        // Continue without blockchain - ticket is already saved
      }
    } else {
      console.warn("⚠️ Blockchain disabled - ticket created offline");
    }

    return res.status(200).json({
      ok: true,
      ticket: { ...saved, attachments },
      jsonHash,
      txHash,
      chainTicketId,
      contract: CONTRACT_ADDRESS,
    });
  } catch (err: any) {
    console.error(err);
    return res
      .status(500)
      .json({ ok: false, error: err?.message ?? "unknown error" });
  }
}
